<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/auth.php';
header('Content-Type: application/json; charset=utf-8');

// Set execution time limit
set_time_limit(30);

if(!isset($_SESSION['user'])){
    http_response_code(401);
    echo json_encode(['ok'=>false, 'error'=>'not authorized']); exit;
}
$admins = ADMIN_LOGINS;
if(!in_array($_SESSION['user']['login'] ?? '', $admins, true)){
    http_response_code(403);
    echo json_encode(['ok'=>false, 'error'=>'forbidden']); exit;
}

if(!isset($_FILES['file'])){
    http_response_code(400);
    echo json_encode(['ok'=>false, 'error'=>'no file']); exit;
}

$f = $_FILES['file'];
if($f['error'] !== UPLOAD_ERR_OK){
    http_response_code(400);
    echo json_encode(['ok'=>false, 'error'=>'upload error ' . $f['error']]); exit;
}
if($f['size'] > MAX_UPLOAD_BYTES){
    http_response_code(400);
    echo json_encode(['ok'=>false, 'error'=>'too big']); exit;
}

// Check if we have enough disk space
$free_space = disk_free_space(UPLOAD_DIR);
if ($free_space < $f['size'] * 2) { // Require 2x space for safety
    http_response_code(507);
    echo json_encode(['ok'=>false, 'error'=>'insufficient disk space']); exit;
}

// Generate a random 6-character prefix (letters, digits, underscore)
$characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_';
$prefix = '';
for ($i = 0; $i < 6; $i++) {
    $prefix .= $characters[mt_rand(0, strlen($characters) - 1)];
}

// Preserve original filename with the random prefix
$fname = $prefix . '_' . $f['name'];
$dest = UPLOAD_DIR . '/' . $fname;

// Check if destination file already exists (very unlikely but possible)
if (file_exists($dest)) {
    http_response_code(500);
    echo json_encode(['ok'=>false, 'error'=>'file name collision']); exit;
}

// Use atomic file move for better reliability
// Check if this is a real uploaded file or a test file
$move_result = false;
if (is_uploaded_file($f['tmp_name'])) {
    // This is a real HTTP uploaded file
    $move_result = move_uploaded_file($f['tmp_name'], $dest);
} else {
    // This is a test file or manually created file
    $move_result = rename($f['tmp_name'], $dest);
}

if(!$move_result){
    // Get more detailed error information
    $error = error_get_last();
    error_log("File move failed: " . print_r($error, true));
    error_log("Source: " . $f['tmp_name']);
    error_log("Destination: " . $dest);
    error_log("Source exists: " . (file_exists($f['tmp_name']) ? 'yes' : 'no'));
    error_log("Destination directory writable: " . (is_writable(UPLOAD_DIR) ? 'yes' : 'no'));
    error_log("Is uploaded file: " . (is_uploaded_file($f['tmp_name']) ? 'yes' : 'no'));
    
    http_response_code(500);
    echo json_encode(['ok'=>false, 'error'=>'move failed: ' . ($error['message'] ?? 'unknown error')]); exit;
}

// Verify the file was moved successfully
if (!file_exists($dest)) {
    http_response_code(500);
    echo json_encode(['ok'=>false, 'error'=>'file not found after move']); exit;
}

// Get MIME type with error handling
$mime = 'application/octet-stream';
if (function_exists('mime_content_type')) {
    $mime = mime_content_type($dest) ?: $mime;
} else if (function_exists('finfo_open')) {
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = finfo_file($finfo, $dest) ?: $mime;
    finfo_close($finfo);
}

// Verify file integrity by checking size
$actual_size = filesize($dest);
if ($actual_size !== $f['size']) {
    // Try to remove the file if size doesn't match
    @unlink($dest);
    http_response_code(500);
    echo json_encode(['ok'=>false, 'error'=>'file size mismatch']); exit;
}

echo json_encode(['ok'=>true, 'file'=>[
    'filename'=>$fname,
    'original_name'=>$f['name'],
    'mime'=>$mime,
    'size'=>intval($f['size'])
]]);